/*
 * Copyright (c) 2006 Oracle.  All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */
#include <linux/kernel.h>
#include <linux/sysctl.h>
#include <linux/proc_fs.h>

#include "ib.h"

static struct ctl_table_header *rds_ib_sysctl_hdr;

/* default to what we hope will be order 0 allocations */
unsigned long rds_ib_sysctl_max_send_wr = PAGE_SIZE / sizeof(struct ib_send_wr);
unsigned long rds_ib_sysctl_max_recv_wr = PAGE_SIZE / sizeof(struct ib_recv_wr);
static unsigned long rds_ib_sysctl_max_wr_min = 1;
/* hardware will fail CQ creation long before this */
static unsigned long rds_ib_sysctl_max_wr_max = (u32)~0;

/* default to rds_ib_sysctl_max_send_wr/4 */
unsigned long rds_ib_sysctl_max_unsig_wrs = PAGE_SIZE / (4 * sizeof(struct ib_send_wr));
static unsigned long rds_ib_sysctl_max_unsig_wr_min = 1;
static unsigned long rds_ib_sysctl_max_unsig_wr_max = PAGE_SIZE / sizeof(struct ib_send_wr);

ctl_table rds_ib_sysctl_table[] = {
	{
		.ctl_name       = 1,
		.procname       = "max_send_wr",
		.data		= &rds_ib_sysctl_max_send_wr,
		.maxlen         = sizeof(unsigned long),
		.mode           = 0644,
		.proc_handler   = &proc_doulongvec_minmax,
		.extra1		= &rds_ib_sysctl_max_wr_min,
		.extra2		= &rds_ib_sysctl_max_wr_max,
	},
	{
		.ctl_name       = 2,
		.procname       = "max_recv_wr",
		.data		= &rds_ib_sysctl_max_recv_wr,
		.maxlen         = sizeof(unsigned long),
		.mode           = 0644,
		.proc_handler   = &proc_doulongvec_minmax,
		.extra1		= &rds_ib_sysctl_max_wr_min,
		.extra2		= &rds_ib_sysctl_max_wr_max,
	},
	{
		.ctl_name       = 3,
		.procname       = "max_unsignaled_wr",
		.data		= &rds_ib_sysctl_max_unsig_wrs,
		.maxlen         = sizeof(unsigned long),
		.mode           = 0644,
		.proc_handler   = &proc_doulongvec_minmax,
		.extra1		= &rds_ib_sysctl_max_unsig_wr_min,
		.extra2		= &rds_ib_sysctl_max_unsig_wr_max,
	},
	{ .ctl_name = 0}
};

static ctl_table rds_ib_sysctl_rds_table[] = {
	{
		.ctl_name	= SYSCTL_NET_RDS_IB,
		.procname	= "ib",
		.data		= NULL,
		.maxlen		= 0,
		.mode		= 0555,
		.child		= rds_ib_sysctl_table,
	},
	{ .ctl_name = 0}
};

static ctl_table rds_ib_sysctl_net_table[] = {
	{
		.ctl_name	= SYSCTL_NET_RDS,
		.procname	= "rds",
		.data		= NULL,
		.maxlen		= 0,
		.mode		= 0555,
		.child		= rds_ib_sysctl_rds_table
	},
	{ .ctl_name = 0}
};

static ctl_table rds_ib_sysctl_root_table[] = {
	{
		.ctl_name	= CTL_NET,
		.procname	= "net",
		.data		= NULL,
		.maxlen		= 0,
		.mode		= 0555,
		.child		= rds_ib_sysctl_net_table
	},
	{ .ctl_name = 0 }
};

void rds_ib_sysctl_exit(void)
{
	if (rds_ib_sysctl_hdr)
		unregister_sysctl_table(rds_ib_sysctl_hdr);
}

int __init rds_ib_sysctl_init(void)
{
	rds_ib_sysctl_hdr = register_sysctl_table(rds_ib_sysctl_root_table, 0);
	if (rds_ib_sysctl_hdr == NULL)
		return -ENOMEM;
	return 0;
}
